/*
 * sethdlc.c
 *
 * Copyright (C) 1999 - 2002 Krzysztof Halasa <khc@pm.waw.pl>
 * Copyright (C) 2003 Piotr Kaczmarzyk <piotr@tahoe.pl>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

#include <sys/sockio.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <sys/errno.h>
#include <net/if.h>
#include <stdarg.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>

static struct ifreq req;
static int argc;
static char **argv;
int	sock;

#define	ADD_PVC		0
#define	DEL_PVC		1
#define	SET_FR		2
#define	SET_G703	3
#define	GET_FR		4
#define GET_G703	5
#define GET_G703_SLOTS	6
#define GET_PROTO	7
#define GET_CISCO	8
#define SET_CISCO	9
#define SET_PPP		10
#define	SET_FR_LMI	11

#define FR_SIG_ANSI	0
#define FR_SIG_Q933A	1
#define FR_SIG_CISCO	2

#define IF_PROTO_CISCO	0
#define IF_PROTO_PPP	1
#define IF_PROTO_FR	2

#define	IF_IFACE_V35		0x01
#define IF_IFACE_E1		0x02

static void error(const char *format, ...) __attribute__ ((noreturn));

static void error(const char *format, ...)
{
	va_list args;

	va_start(args, format);
	fprintf(stderr, "%s: ", req.ifr_name);
	vfprintf(stderr, format, args);
	va_end(args);
	exit(1);
}

typedef struct {
	const char *name;
	const unsigned int value;
} parsertab;



static int checkkey(const char* name)
{
	if (argc < 1)
		return -1;	/* no enough parameters */

	if (strcmp(name, argv[0]))
		return -1;
	argc--;
	argv++;
	return 0;
}



static int checktab(parsertab *tab, unsigned int *value)
{
	int i;

	if (argc < 1)
		return -1;	/* no enough parameters */
	
	for (i = 0; tab[i].name; i++)
		if (!strcmp(tab[i].name, argv[0])) {
			argc--;
			argv++;
			*value = tab[i].value;
			return 0;
		}

	return -1;		/* Not found */
}



static const char* tabstr(unsigned int value, parsertab *tab,
			  const char* unknown)
{
	int i;
	for (i = 0; tab[i].name; i++)
		if (tab[i].value == value)
			return tab[i].name;

	return unknown;		/* Not found */
}



static unsigned int match(const char* name, unsigned int *value,
			  unsigned int minimum, unsigned int maximum)
{
	char test;

	if (argc < 1)
		return -1;	/* no enough parameters */

	if (name) {
		if (strcmp(name, argv[0]))
			return -1;
		argc--;
		argv++;
	}

	if (argc < 1)
		error("Missing parameter\n");

	if (sscanf(argv[0], "%u%c", value, &test) != 1)
		error("Invalid parameter: %s\n", argv[0]);

	if ((*value > maximum) || (*value < minimum))
		error("Parameter out of range [%u - %u]: %u\n",
		      minimum, maximum, *value);

	argc--;
	argv++;
	return 0;
}


static parsertab ifaces[] = {{ "v35", IF_IFACE_V35 },
			     { "e1", IF_IFACE_E1 },
			     { NULL, 0 }};

static parsertab protos[] = {{ "cisco", IF_PROTO_CISCO},
			     { "fr", IF_PROTO_FR},
			     { "ppp", IF_PROTO_PPP},
			     { NULL, 0 }};

static parsertab lmi[] = {{ "ansi", FR_SIG_ANSI },
			  { "ccitt", FR_SIG_Q933A },
			  { "cisco", FR_SIG_CISCO },
			  { NULL, 0 }};

static void set_pvc(void)
{
	char *op = argv[0];
	parsertab ops[] = {{ "create", ADD_PVC },
			   { "delete", DEL_PVC },
			   { NULL, 0 }};
	u_int	type;
	u_int	dlci;

	if (checktab(ops, &type))
		return;

	if (match(NULL, &dlci, 0, 1023))
		return;

	if (argc != 0)
		return;

	req.ifr_phys = (type << 24) | dlci;

	if (ioctl(sock, SIOCGPRIVATE_0, &req))
		error("Unable to %s PVC: %s\n", op, strerror(errno));
	exit(0);
}

static int check_slot(char slot)
{
	if ((slot < 1) || (slot > 32)) {
		printf("Slots have to be between 1 and 32\n");
		return 1;
	}
	return 0;
}

static unsigned int get_slot_map()
{
	char	*cp, *c, *c2, slot, slot_end;
	unsigned int slots;
	
	if (argc < 1) {
		printf("Required parameter missing\n");
		return 0;
	}
	
	if (!strcmp("all", argv[0])) {
		argc--;
		argv++;
		return 0xffffffff;
	}

	cp = c = argv[0];
	slots = 0;
	while (1) {
		char	out;
				
		out = 0;
		switch (*cp) {
			case ',':
			case 0:
				if (*cp == 0) out = 1;	// Koniec linijki
				*cp++ = 0;
				slot = atoi(c);
				if (check_slot(slot))
					exit(1);
				slots |= 1 << (slot - 1);
				c = cp;
				break;
			case '-':
				/* Zakres */
				*cp++ = 0;
				c2 = cp;
				while (1) {
					if ((*cp == ',') || (*cp == 0)) {
						if (*cp == 0) out = 1;	// Koniec linijki
						*cp++ = 0;
						slot = atoi(c);
						slot_end = atoi(c2);
						if (check_slot(slot) || check_slot(slot_end))
							exit(1);
						if (slot_end < slot) {
							printf("The ending slot should be higher than the starting one\n");
							exit(1);
						}
						do {
							slots |= 1 << (slot - 1);
							slot++;
						} while (slot <= slot_end);
						c = cp;
						break;
					}
					cp++;
				}
				break;
			default:
				cp++;
		}
		if (out) break;
	}
	argc--;
	argv++;
	if (slots & 1) {
		printf("Slot 1 is reserved - use 'slots all' to enable unframed mode\n");
		exit(1);
	}
	return slots;
}

void check_err()
{
	if (errno != EINVAL) {
		printf("unable to get interface information: %s\n",
		       strerror(errno));
		close(sock);
		exit(1);
	}
}

u_int	interface, proto, g703_slots, g703_crc4, g703_egl;
u_int	fr_lmi, fr_dce, fr_t391, fr_t392, fr_n391, fr_n392, fr_n393;
u_int	cisco_timeout, cisco_interval;

static void set_iface(void)
{
	int orig_argc = argc;

	req.ifr_phys = (GET_PROTO << 24);
	if (ioctl(sock, SIOCGPRIVATE_0, &req)) check_err();
	interface = req.ifr_phys >> 16;
	proto = req.ifr_phys & 0xffff;

	if (interface == IF_IFACE_E1) {
		req.ifr_phys = (GET_G703 << 24);
		if (ioctl(sock, SIOCGPRIVATE_0, &req)) check_err();
		g703_crc4 = (req.ifr_phys >> 8) & 0xff;
		g703_egl = req.ifr_phys & 0xff;
	
		req.ifr_phys = (GET_G703_SLOTS << 24);
		if (ioctl(sock, SIOCGPRIVATE_0, &req)) check_err();
		g703_slots = req.ifr_phys;
	} else {
		return;
	}

	while (argc > 0) {
		/* slotmap goes here */
		if (!checkkey("slotmap")) {
			g703_slots = get_slot_map();
			if (!g703_slots) return;
			continue;
		}

		if (!checkkey("short")) {
			g703_egl = 0;
			continue;
		}

		if (!checkkey("long")) {
			g703_egl = 1;
			continue;
		}

		if (!checkkey("crc4")) {
			if (!checkkey("on")) {
				g703_crc4 = 1;
				continue;
			}
			if (!checkkey("off")) {
				g703_crc4 = 0;
				continue;
			}
		}

		if (orig_argc == argc)
			return;	/* not an iface definition */
		error("Invalid parameter: %s\n", argv[0]);
	}

	req.ifr_phys = (SET_G703 << 24) | (g703_crc4 << 8) | g703_egl;
	if (ioctl(sock, SIOCGPRIVATE_0, &req))
		error("Unable to set interface information: %s\n",
		      strerror(errno));

	req.ifr_phys = g703_slots;
	if (ioctl(sock, SIOCGPRIVATE_1, &req))
		error("Unable to set interface information: %s\n",
		      strerror(errno));

	exit(0);
}



static void set_proto_fr(void)
{
	int lmi_type = -1, debug = 0;

	while (argc > 0) {
		if (debug == 0) {
			if (!checkkey("debug")) {
				debug = 1;
				continue;
			}
		}
		if (lmi_type == -1)
			if (!checkkey("lmi")) {
				if (!checktab(lmi, &lmi_type))
					continue;
				error("Invalid LMI type: %s\n",
				      argv[0]);
			}

		if (!fr_t391)
			if (!match("t391", &fr_t391,
				   1, 1000))
				continue;
		if (!fr_n391)
			if (!match("n391", &fr_n391,
				   1, 1000))
				continue;
		if (!fr_n392)
			if (!match("n392", &fr_n392,
				   1, 1000))
				continue;
		error("Invalid parameter: %s\n", argv[0]);
	}

	 /* polling verification timer*/
	if (!fr_t391) fr_t391 = 5;
	/* full status polling counter*/
	if (!fr_n391) fr_n391 = 5;
	/* error threshold */
	if (!fr_n392) fr_n392 = 3;

	if (lmi_type == -1)
		fr_lmi = FR_SIG_ANSI;
	else
		fr_lmi = lmi_type;

	req.ifr_phys = (SET_FR << 24) | (fr_n391 << 16) | (fr_n392 << 8) | fr_t391;
	if (ioctl(sock, SIOCGPRIVATE_0, &req))
		error("Unable to set FR protocol information: %s\n",
		      strerror(errno));
	req.ifr_phys = (SET_FR_LMI << 24) | (debug << 8) | fr_lmi;
	if (ioctl(sock, SIOCGPRIVATE_0, &req))
		error("Unable to set FR protocol information: %s\n",
		      strerror(errno));
}

static void set_proto_cisco(void)
{
	while (argc > 0) {
		if (!cisco_interval)
			if (!match("interval", &cisco_interval,
				   1, 100))
				continue;
		if (!cisco_timeout)
			if (!match("timeout", &cisco_timeout,
				   1, 100))
				continue;

		error("Invalid parameter: %s\n",
		      argv[0]);
	}

	if (!cisco_interval)
		cisco_interval = 10;
	if (!cisco_timeout)
		cisco_timeout = 25;

	req.ifr_phys = (SET_CISCO << 24) | (cisco_interval << 8) | cisco_timeout;

	if (ioctl(sock, SIOCGPRIVATE_0, &req))
		error("Unable to set Cisco HDLC protocol information: %s\n",
		      strerror(errno));
}

static void set_proto(void)
{
	if (checktab(protos, &proto))
		return;

	switch(proto) {
	case IF_PROTO_CISCO: set_proto_cisco(); break;
	case IF_PROTO_FR: set_proto_fr(); break;
	case IF_PROTO_PPP:
		req.ifr_phys = (SET_PPP << 24);
		if (ioctl(sock, SIOCGPRIVATE_0, &req))
			error("Unable to set PPP protocol information: %s\n", strerror(errno));
		break;

	default: error("Unknown protocol %u\n", proto);
	}

	if (argc > 0)
		error("Unexpected parameter: %s\n", argv[0]);

	close(sock);
	exit(0);
}


static void display_slots(u_int slots, u_char *tmp, u_char *bin, u_char *n)
{
	u_char	tmp2[10], i, zakr = 255, zakr_begin = 0, comma = 0, slot;

	if (slots == 0xffffffff) {
		sprintf(tmp, "all");
		sprintf(bin, "11111111111111111111111111111111");
		*n = 32;
		return;
	}
	tmp[0] = 0;
	for (i=0; i<32; i++) {
		slot = ((slots & (1 << i)) != 0);
		if (slot) {
			if (zakr == 255) {
				zakr = i;
				zakr_begin = i;
				if (comma) strcat(tmp,",");
				sprintf(tmp2, "%d", i+1);
				strcat(tmp, tmp2);
				comma = 1;
			} else {
				if (zakr != i-1) {
					if (zakr == zakr_begin)
						/* Zakres byl jednopozycyjny */
						sprintf(tmp2, ",%d", i+1);
					else
						sprintf(tmp2, "-%d,%d", zakr+1, i+1);
					strcat(tmp, tmp2);
					zakr_begin = i;
					comma = 1;
				}
				zakr = i;
			}
		}
		if (i == 31) {	/* Ostatni slot */
			if (zakr != 255) {
				tmp2[0] = 0;
				/* Dokoncz rozpoczety zakres */
				if (slot && (zakr_begin != 31))
					sprintf(tmp2, "-32");
				else
					if (zakr != zakr_begin)
						sprintf(tmp2, "-%d", zakr+1);
				strcat(tmp, tmp2);
			}
		}
		if (n) (*n) += slot;
		if (bin) bin[i] = '0' + slot;
		if (bin) bin[0] = 's';	// Slot 1 - synchronizacja
		if (bin) bin[32] = 0;
	}
}

static void show_port(void)
{
	const char *s;
	char tmp[128], bin[128];
	u_char	n;

	printf("%s: ", req.ifr_name);

	req.ifr_phys = (GET_PROTO << 24);
	if (ioctl(sock, SIOCGPRIVATE_0, &req)) check_err();
	interface = req.ifr_phys >> 16;
	proto = req.ifr_phys & 0xffff;
	
	if (proto == IF_PROTO_FR) {
		req.ifr_phys = (GET_FR << 24);
		if (ioctl(sock, SIOCGPRIVATE_0, &req)) check_err();
		fr_lmi = (req.ifr_phys >> 24) & 0xff;
		fr_n391 = (req.ifr_phys >> 16) & 0xff;
		fr_n392 = (req.ifr_phys >> 8) & 0xff;
		fr_t391 = req.ifr_phys & 0xff;
	}
	
	if (proto == IF_PROTO_CISCO) {
		req.ifr_phys = (GET_CISCO << 24);
		if (ioctl(sock, SIOCGPRIVATE_0, &req)) check_err();
		cisco_interval = (req.ifr_phys >> 8) & 0xff;
		cisco_timeout = req.ifr_phys & 0xff;
	}
	
	if (interface == IF_IFACE_E1) {
		req.ifr_phys = (GET_G703 << 24);
		if (ioctl(sock, SIOCGPRIVATE_0, &req)) check_err();
		g703_crc4 = (req.ifr_phys >> 8) & 0xff;
		g703_egl = req.ifr_phys & 0xff;
	
		req.ifr_phys = (GET_G703_SLOTS << 24);
		if (ioctl(sock, SIOCGPRIVATE_0, &req)) check_err();
		g703_slots = req.ifr_phys;
	}
	
	/* Get and print physical interface settings */
	s = tabstr(interface, ifaces, NULL);

	if (!s)
		printf("\n\tunknown interface 0x%x\n", interface);
	else {
		if (*s)
			printf("\n\tinterface %s\n", s);

		if (interface == IF_IFACE_E1) {
			printf("\tslotmap ");
			n = 0;
			display_slots(g703_slots, tmp, bin, &n);
			printf("%s (%s, %d kb/s)\n", tmp, bin, n*64);
			printf("\tcrc4 %s\n\t%s\n", g703_crc4 ? "on" : "off", g703_egl ? "long" : "short");
		}
	}

	/* Get and print protocol settings */
	do {
		printf("\t");

		switch(proto) {
		case IF_PROTO_FR:
			printf("protocol fr lmi %s",
			       tabstr(fr_lmi, lmi, "unknown"));
			printf(" t391 %u n391 %u n392 %u\n",
			       fr_t391,
			       fr_n391,
			       fr_n392);
			break;

		case IF_PROTO_CISCO:
			printf("protocol cisco\n"); // interval %u timeout %u\n",
			 //      cisco_interval,
			 //      cisco_timeout);
			break;

		case IF_PROTO_PPP:
			printf("protocol ppp\n");
			break;

		default:
			printf("unknown protocol %u\n", proto);
		}
	}while(0);

	close(sock);
	exit(0);
}

static void usage(void)
{
	fprintf(stderr, "sethdlc version 1.15t\n"
		"Copyright (C) 2000 - 2003 Krzysztof Halasa <khc@pm.waw.pl>\n"
		"Copyright (C) 2003 Piotr Kaczmarzyk <piotr@tahoe.pl>\n"
		"\n"
		"Usage: sethdlc INTERFACE "
		"[slotmap SLOTMAP] [crc4 on | off] [short | long]\n"
		"       sethdlc INTERFACE [PROTOCOL]\n"
		"       sethdlc INTERFACE create | delete"
		" DLCI\n"
		"\n"
		"PROTOCOL := cisco |\n" //"[interval val] [timeout val] |\n"
		"            fr [lmi LMI] [debug] |\n"
		"            ppp\n"
		"\n"
		"LMI := ansi [LMI_SPEC] | ccitt [LMI_SPEC] | cisco [LMI_SPEC]\n"
		"LMI_SPEC := [t391 val] [n391 val] [n392 val]\n");
	exit(0);
}

int main(int arg_c, char *arg_v[])
{
	argc = arg_c;
	argv = arg_v;

	if (argc <= 1)
		usage();
  
	sock = socket(PF_INET, SOCK_DGRAM, IPPROTO_IP);
	if (sock < 0)
		error("Unable to create socket: %s\n", strerror(errno));
  
	strcpy(req.ifr_name, argv[1]); /* Device name */

	if (argc == 2)
		show_port();

	argc -= 2;
	argv += 2;

	set_iface();
	set_proto();
	set_pvc();

	close(sock);
	usage();
	exit(0);
}
